import { FormEvent, useState } from "react";
import { signIn, useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";

export default function SignupPage() {
  const { t } = useLanguage();
  const { data: session } = useSession();
  const router = useRouter();

  const [email, setEmail] = useState("");
  const [name, setName] = useState("");
  const [password, setPassword] = useState("");
  const [confirm, setConfirm] = useState("");
  const [err, setErr] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  if (session?.user) {
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='card max-w-md w-full p-8 text-center'>
          <h1 className='text-2xl font-semibold text-foreground mb-2'>
            {t("signup.alreadySignedIn") || "Already signed in"}
          </h1>
          <p className='text-muted mb-6'>
            {t("common.loggedInAs") || "You are logged in as"}{" "}
            {session.user.email}
          </p>
          <Link
            href='/'
            className='btn-primary w-full inline-flex items-center justify-center'
          >
            {t("common.goHome") || "Go home"}
          </Link>
        </div>
      </div>
    );
  }

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);

    if (password !== confirm) {
      setErr(t("signup.passwordsDoNotMatch") || "Passwords do not match");
      return;
    }

    setLoading(true);
    const res = await fetch("/api/auth/register", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ email, name, password }),
    });
    const json = await res.json();
    setLoading(false);

    if (!json.ok) {
      setErr(json.error || "Sign up failed");
      return;
    }

    const login = await signIn("credentials", {
      redirect: false,
      email,
      password,
    });
    if (login?.ok) router.push("/");
    else router.push("/login");
  };

  return (
    <div className='min-h-screen bg-sidebar relative'>
      {/* Background decoration */}
      <div
        aria-hidden
        className='pointer-events-none absolute inset-0 -z-10 overflow-hidden'
      >
        <div className='absolute -top-24 -right-20 size-80 rounded-full bg-sidebar-accent/40 blur-3xl' />
        <div className='absolute -bottom-24 -left-20 size-80 rounded-full bg-primary/20 blur-3xl' />
      </div>

      <div className='mx-auto max-w-6xl px-4 py-8 md:py-12 lg:py-20'>
        {/* Mobile/Tablet Header */}
        <div className='md:hidden mb-8 text-center'>
          <h1 className='text-2xl font-semibold text-foreground leading-tight mb-2'>
            {t("signup.title")}
          </h1>
          <p className='text-muted text-sm'>{t("signup.subtitle")}</p>
        </div>

        <div className='grid grid-cols-1 md:grid-cols-2 gap-8 md:gap-12 items-center'>
          {/* Left copy (Desktop/Tablet) */}
          <div className='hidden md:block'>
            <h1 className='text-2xl md:text-3xl xl:text-4xl font-semibold text-foreground leading-tight mb-3'>
              {t("signup.title")}
            </h1>
            <p className='text-muted max-w-prose'>
              {t("signup.subtitleDesktop")}
            </p>
          </div>

          {/* Right card */}
          <div className='w-full max-w-md mx-auto md:mx-0 md:ml-auto'>
            <div className='card'>
              <div className='px-6 sm:px-8 pt-8 pb-4 text-center'>
                <h2 className='text-2xl font-semibold text-foreground'>
                  {t("signup.signUp")}
                </h2>
                <p className='text-muted mt-1'>{t("signup.createAccount")}</p>
              </div>

              <form
                onSubmit={onSubmit}
                className='px-6 sm:px-8 pb-8 space-y-5'
                noValidate
              >
                {err && (
                  <div
                    role='alert'
                    aria-live='polite'
                    className='rounded-lg border border-red-200 bg-red-50 px-4 py-3 text-sm text-red-800'
                  >
                    {err}
                  </div>
                )}

                <div className='space-y-2'>
                  <label
                    htmlFor='email'
                    className='block text-sm font-medium text-foreground'
                  >
                    {t("login.email")}
                  </label>
                  <input
                    id='email'
                    type='email'
                    required
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    className='input-field'
                    placeholder='name@hospital.org'
                    autoComplete='email'
                  />
                </div>

                <div className='space-y-2'>
                  <label
                    htmlFor='name'
                    className='block text-sm font-medium text-foreground'
                  >
                    {t("common.name")}
                  </label>
                  <input
                    id='name'
                    type='text'
                    value={name}
                    onChange={(e) => setName(e.target.value)}
                    className='input-field'
                    placeholder={t("common.yourName") || "Your name"}
                    autoComplete='name'
                  />
                </div>

                <div className='space-y-2'>
                  <label
                    htmlFor='password'
                    className='block text-sm font-medium text-foreground'
                  >
                    {t("login.password")} (min 6)
                  </label>
                  <input
                    id='password'
                    type='password'
                    required
                    minLength={6}
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    className='input-field'
                    placeholder='••••••••'
                    autoComplete='new-password'
                  />
                </div>

                <div className='space-y-2'>
                  <label
                    htmlFor='confirm'
                    className='block text-sm font-medium text-foreground'
                  >
                    {t("signup.confirmPassword")}
                  </label>
                  <input
                    id='confirm'
                    type='password'
                    required
                    minLength={6}
                    value={confirm}
                    onChange={(e) => setConfirm(e.target.value)}
                    className='input-field'
                    placeholder='••••••••'
                    autoComplete='new-password'
                  />
                </div>

                <button
                  type='submit'
                  disabled={loading}
                  className='btn-primary w-full flex items-center justify-center disabled:opacity-50 disabled:cursor-not-allowed'
                >
                  {loading ? `${t("signup.creating")}…` : t("signup.signUp")}
                </button>
              </form>

              <div className='px-6 sm:px-8 pb-6 pt-3 border-t border-card text-center'>
                <p className='text-sm text-muted'>
                  {t("signup.alreadyHaveAccount")}{" "}
                  <Link
                    href='/login'
                    className='text-primary hover:opacity-80 font-medium'
                  >
                    {t("nav.signIn")}
                  </Link>
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
